<?php
/*
    *****************************************************************************
    * MEGA Digital Space - by MEGA Digital Studio                               *
    * ------------------------------------------------------------------------- *
    * CMS Website:  https://megadigital.space                                   *
    * License:  https://megadigital.space/license                               *
    * Owner:  Giorgi Basharuli                                                  *
    * Contact:  online@megadigital.studio                                       *
    * MEGA Digital Studio – Official Development Studio for MEGA Digital Space  *
    * Website:  https://megadigital.studio                                      *
    * ------------------------------------------------------------------------- *
    * Description: This file is a part of the MEGA Digital Space CMS system.    *
    * Unauthorized distribution, modification, or commercial use is strictly -  *
    * prohibited without prior written permission.                              *
    * ------------------------------------------------------------------------- *
    * Copyright (c) 2023-2025 MEGA Digital Studio.                              *
    * All rights reserved.                                                      *
    *****************************************************************************
*/

require_once($_SERVER["DOCUMENT_ROOT"] . '/engine/data/config.php');
require_once($_SERVER["DOCUMENT_ROOT"] . "/engine/data/dbconfig.php");
//require_once(SYSTEM_DIR. '/plugins/reserved_fm_names.php');

function getFormattedPermissions($filePath) {
    $perms = fileperms($filePath);

    // Определяем тип файла
    switch ($perms & 0xF000) {
        case 0xC000: $type = 's'; break; // Сокет
        case 0xA000: $type = 'l'; break; // Символическая ссылка
        case 0x8000: $type = '-'; break; // Обычный файл
        case 0x6000: $type = 'b'; break; // Блочное устройство
        case 0x4000: $type = 'd'; break; // Каталог
        case 0x2000: $type = 'c'; break; // Символьное устройство
        case 0x1000: $type = 'p'; break; // Канал FIFO
        default:     $type = 'u'; break; // Неизвестный тип
    }

    // Права для владельца
    $owner = (($perms & 0x0100) ? 'r' : '-') .
        (($perms & 0x0080) ? 'w' : '-') .
        (($perms & 0x0040) ? (($perms & 0x0800) ? 's' : 'x') : (($perms & 0x0800) ? 'S' : '-'));

    // Права для группы
    $group = (($perms & 0x0020) ? 'r' : '-') .
        (($perms & 0x0010) ? 'w' : '-') .
        (($perms & 0x0008) ? (($perms & 0x0400) ? 's' : 'x') : (($perms & 0x0400) ? 'S' : '-'));

    // Права для остальных
    $others = (($perms & 0x0004) ? 'r' : '-') .
        (($perms & 0x0002) ? 'w' : '-') .
        (($perms & 0x0001) ? (($perms & 0x0200) ? 't' : 'x') : (($perms & 0x0200) ? 'T' : '-'));

    return $type . $owner . $group . $others;
}
function getReservedFMNames()
{
    return [".git", ".gitignore", ".osp", "docs", "demo", "documents", ".idea", ".trash", "templates_c"];
}
function getReservedFMExtensions()
{
    return ["md", "rar", "zip"];
}
function searchFiles($path, $query) {
    if (!is_dir($path)) {
        return [];
    }

    // Если query пустое, просто вернем все файлы и папки в текущей директории
    if (empty($query)) {
        $items = scandir($path);
        $result = [];

        foreach ($items as $item) {
            if ($item === '.' || $item === '..') continue;

            $fullPath = $path . '/' . $item;
            // Пропускаем метаданные
            $extension = pathinfo($_SERVER["DOCUMENT_ROOT"] . "/" . $fullPath, PATHINFO_EXTENSION);
            if ($extension == "metadata") {
                continue;
            }

            $reservedNames = getReservedFMNames();
            $reservedExtensions = getReservedFMExtensions();
            $continue = false;
            foreach ($reservedNames as $reservedName) {
                if ($item == $reservedName) {
                    $continue = true;
                }
            }
            foreach ($reservedExtensions as $reservedExtension) {
                if ($extension == $reservedExtension) {
                    $continue = true;
                }
            }
            if ($continue) {
                continue;
            }

            // Получаем права файла
            $permissions = getFormattedPermissions($fullPath);

            // Получаем размер файла или директории
            if (is_file($fullPath)) {
                $fileSize = filesize($fullPath);
                $fileSizeInMB = $fileSize / 1024 / 1024;
                $fileSizeInKB = $fileSize / 1024;
            } else {
                $fileSizeInMB = 0;
                $fileSizeInKB = 0;
                $fileSize = 0;
            }

            $result[] = [
                'extension' => $extension,
                'size' => $fileSizeInMB > 1
                    ? number_format($fileSizeInMB, 2) . " MB"
                    : ($fileSizeInKB >= 1
                        ? number_format($fileSizeInKB, 2) . " KB"
                        : ($fileSize > 0
                            ? $fileSize . " байт"
                            : "Папка")),
                'full_path' => $fullPath,
                'name' => $item,
                'is_dir' => is_dir($fullPath),
                'permissions' => $permissions,
                'normal_path' => str_replace($_SERVER["DOCUMENT_ROOT"] . "/", "", $fullPath) != "uploads"
                    ? str_replace($_SERVER["DOCUMENT_ROOT"] . "/", "", $fullPath)
                    : $item
            ];
        }

        return $result;
    }

    // Если query не пустое, выполняем рекурсивный поиск
    $items = scandir($path);
    $result = [];

    foreach ($items as $item) {
        // Пропускаем текущую и родительскую директории
        if ($item === '.' || $item === '..') continue;

        $fullPath = $path . '/' . $item;

        // Пропускаем метаданные
        $extension = pathinfo($_SERVER["DOCUMENT_ROOT"] . "/" . $fullPath, PATHINFO_EXTENSION);
        if ($extension == "metadata") {
            continue;
        }

        // Получаем права файла
        $permissions = getFormattedPermissions($fullPath);

        // Получаем размер файла или директории
        if (is_file($fullPath)) {
            $fileSize = filesize($fullPath);
            $fileSizeInMB = $fileSize / 1024 / 1024;
            $fileSizeInKB = $fileSize / 1024;
        } else {
            $fileSizeInMB = 0;
            $fileSizeInKB = 0;
            $fileSize = 0;
        }

        // Добавляем файл в результат, если имя файла содержит поисковый запрос
        if (stripos($item, $query) !== false) {
            $result[] = [
                'extension' => $extension,
                'size' => $fileSizeInMB > 1
                    ? number_format($fileSizeInMB, 2) . " MB"
                    : ($fileSizeInKB >= 1
                        ? number_format($fileSizeInKB, 2) . " KB"
                        : ($fileSize > 0
                            ? $fileSize . " байт"
                            : "Папка")),
                'full_path' => $fullPath,
                'name' => $item,
                'is_dir' => is_dir($fullPath),
                'permissions' => $permissions,
                'normal_path' => str_replace($_SERVER["DOCUMENT_ROOT"] . "/", "", $fullPath) != "uploads"
                    ? str_replace($_SERVER["DOCUMENT_ROOT"] . "/", "", $fullPath)
                    : $item
            ];
        }

        // Если это директория, выполняем рекурсивный поиск в этой директории
        if (is_dir($fullPath)) {
            $result = array_merge($result, searchFiles($fullPath, $query));
        }
    }
    return $result;
}

// Получаем данные из POST-запроса
$data = json_decode(file_get_contents('php://input'), true);
$query = $data['query'] ?? '';

// Путь к директории, в которой нужно искать
$rootDir = $data["currentPath"] == "" ? $_SERVER["DOCUMENT_ROOT"] : $_SERVER["DOCUMENT_ROOT"]."/".$data["currentPath"]; // Укажите путь к вашему каталогу

// Выполняем поиск и возвращаем результаты
$results = searchFiles($rootDir, $query);
$html = "";
$filesCount = 0;
$folderCount = 0;
$totalSizeM = 0;
$directoryStats = [];

// Обрабатываем результаты поиска
foreach ($results as $result) {
    $html .= renderFileRow($result);
    $directoryStats[] = getDirectoryStats($result["full_path"]);
}

foreach ($directoryStats as $stats) {
    $totalSizeM += $stats["total_size"];
    $folderCount += $stats["folders"];
    $filesCount += $stats["files"];
}

echo json_encode([
    "html" => $html,
    "pathWithoutLast" => "",
    "directoryStats" => [
        "total_size" => round($totalSizeM),
        "files" => $filesCount,
        "folders" => $folderCount,
    ],
]);


// Функция для получения статистики директории
function getDirectoryStats($directory) {
    if (!is_dir($directory)) {
        return ['folders' => 0, 'files' => 0, 'total_size' => 0];
    }

    $fileCount = 0;
    $folderCount = 0;
    $totalSize = 0;

    try {
        $directoryIterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory, FilesystemIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($directoryIterator as $fileinfo) {
            if ($fileinfo->isFile()) {
                $fileCount++;
                $totalSize += $fileinfo->getSize();
            } elseif ($fileinfo->isDir()) {
                $folderCount++;
            }
        }
    } catch (Exception $e) {
        error_log("Ошибка при анализе директории: " . $e->getMessage());
    }

    $totalSizeMB = $totalSize / 1024 / 1024;

    return [
        'folders' => $folderCount,
        'files' => $fileCount,
        'total_size' => number_format($totalSizeMB, 2)
    ];
}

function renderFileRow($file) {
    $icons = [
        'git' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-github" viewBox="0 0 16 16">
                                                        <path d="M8 0C3.58 0 0 3.58 0 8c0 3.54 2.29 6.53 5.47 7.59.4.07.55-.17.55-.38 0-.19-.01-.82-.01-1.49-2.01.37-2.53-.49-2.69-.94-.09-.23-.48-.94-.82-1.13-.28-.15-.68-.52-.01-.53.63-.01 1.08.58 1.23.82.72 1.21 1.87.87 2.33.66.07-.52.28-.87.51-1.07-1.78-.2-3.64-.89-3.64-3.95 0-.87.31-1.59.82-2.15-.08-.2-.36-1.02.08-2.12 0 0 .67-.21 2.2.82.64-.18 1.32-.27 2-.27s1.36.09 2 .27c1.53-1.04 2.2-.82 2.2-.82.44 1.1.16 1.92.08 2.12.51.56.82 1.27.82 2.15 0 3.07-1.87 3.75-3.65 3.95.29.25.54.73.54 1.48 0 1.07-.01 1.93-.01 2.2 0 .21.15.46.55.38A8.01 8.01 0 0 0 16 8c0-4.42-3.58-8-8-8"/>
                                                    </svg>',
        'gitignore' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-github" viewBox="0 0 16 16">
                                                        <path d="M8 0C3.58 0 0 3.58 0 8c0 3.54 2.29 6.53 5.47 7.59.4.07.55-.17.55-.38 0-.19-.01-.82-.01-1.49-2.01.37-2.53-.49-2.69-.94-.09-.23-.48-.94-.82-1.13-.28-.15-.68-.52-.01-.53.63-.01 1.08.58 1.23.82.72 1.21 1.87.87 2.33.66.07-.52.28-.87.51-1.07-1.78-.2-3.64-.89-3.64-3.95 0-.87.31-1.59.82-2.15-.08-.2-.36-1.02.08-2.12 0 0 .67-.21 2.2.82.64-.18 1.32-.27 2-.27s1.36.09 2 .27c1.53-1.04 2.2-.82 2.2-.82.44 1.1.16 1.92.08 2.12.51.56.82 1.27.82 2.15 0 3.07-1.87 3.75-3.65 3.95.29.25.54.73.54 1.48 0 1.07-.01 1.93-.01 2.2 0 .21.15.46.55.38A8.01 8.01 0 0 0 16 8c0-4.42-3.58-8-8-8"/>
                                                    </svg>',
        'php' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-filetype-php" viewBox="0 0 16 16">
                                                        <path fill-rule="evenodd" d="M14 4.5V14a2 2 0 0 1-2 2h-1v-1h1a1 1 0 0 0 1-1V4.5h-2A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v9H2V2a2 2 0 0 1 2-2h5.5zM1.6 11.85H0v3.999h.791v-1.342h.803q.43 0 .732-.173.305-.175.463-.474a1.4 1.4 0 0 0 .161-.677q0-.375-.158-.677a1.2 1.2 0 0 0-.46-.477q-.3-.18-.732-.179m.545 1.333a.8.8 0 0 1-.085.38.57.57 0 0 1-.238.241.8.8 0 0 1-.375.082H.788V12.48h.66q.327 0 .512.181.185.182.185.522m4.48 2.666V11.85h-.79v1.626H4.153V11.85h-.79v3.999h.79v-1.714h1.682v1.714zm.703-3.999h1.6q.433 0 .732.179.3.175.46.477.158.302.158.677t-.161.677q-.159.299-.463.474a1.45 1.45 0 0 1-.733.173H8.12v1.342h-.791zm2.06 1.714a.8.8 0 0 0 .084-.381q0-.34-.184-.521-.184-.182-.513-.182h-.66v1.406h.66a.8.8 0 0 0 .375-.082.57.57 0 0 0 .237-.24Z"/>
                                                    </svg>',
        'js' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-filetype-js" viewBox="0 0 16 16">
                                                        <path fill-rule="evenodd" d="M14 4.5V14a2 2 0 0 1-2 2H8v-1h4a1 1 0 0 0 1-1V4.5h-2A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v9H2V2a2 2 0 0 1 2-2h5.5zM3.186 15.29a1.2 1.2 0 0 1-.111-.449h.765a.58.58 0 0 0 .255.384q.105.073.249.114.143.041.319.041.246 0 .413-.07a.56.56 0 0 0 .255-.193.5.5 0 0 0 .085-.29.39.39 0 0 0-.153-.326q-.151-.12-.462-.193l-.619-.143a1.7 1.7 0 0 1-.539-.214 1 1 0 0 1-.351-.367 1.1 1.1 0 0 1-.123-.524q0-.366.19-.639.19-.272.528-.422.336-.15.776-.149.457 0 .78.152.324.153.5.41.18.255.2.566h-.75a.56.56 0 0 0-.12-.258.6.6 0 0 0-.247-.181.9.9 0 0 0-.369-.068q-.325 0-.513.152a.47.47 0 0 0-.184.384q0 .18.143.3a1 1 0 0 0 .405.175l.62.143q.327.075.566.211.24.136.375.358t.135.56q0 .37-.188.656a1.2 1.2 0 0 1-.539.439q-.351.158-.858.158-.381 0-.665-.09a1.4 1.4 0 0 1-.478-.252 1.1 1.1 0 0 1-.29-.375m-3.104-.033A1.3 1.3 0 0 1 0 14.791h.765a.6.6 0 0 0 .073.27.5.5 0 0 0 .454.246q.285 0 .422-.164.138-.165.138-.466v-2.745h.79v2.725q0 .66-.357 1.005-.354.345-.984.345a1.6 1.6 0 0 1-.569-.094 1.15 1.15 0 0 1-.407-.266 1.1 1.1 0 0 1-.243-.39"/>
                                                    </svg>',
        'json' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-filetype-json" viewBox="0 0 16 16">
                                                        <path fill-rule="evenodd" d="M14 4.5V11h-1V4.5h-2A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v9H2V2a2 2 0 0 1 2-2h5.5zM4.151 15.29a1.2 1.2 0 0 1-.111-.449h.764a.58.58 0 0 0 .255.384q.105.073.25.114.142.041.319.041.245 0 .413-.07a.56.56 0 0 0 .255-.193.5.5 0 0 0 .084-.29.39.39 0 0 0-.152-.326q-.152-.12-.463-.193l-.618-.143a1.7 1.7 0 0 1-.539-.214 1 1 0 0 1-.352-.367 1.1 1.1 0 0 1-.123-.524q0-.366.19-.639.192-.272.528-.422.337-.15.777-.149.456 0 .779.152.326.153.5.41.18.255.2.566h-.75a.56.56 0 0 0-.12-.258.6.6 0 0 0-.246-.181.9.9 0 0 0-.37-.068q-.324 0-.512.152a.47.47 0 0 0-.185.384q0 .18.144.3a1 1 0 0 0 .404.175l.621.143q.326.075.566.211a1 1 0 0 1 .375.358q.135.222.135.56 0 .37-.188.656a1.2 1.2 0 0 1-.539.439q-.351.158-.858.158-.381 0-.665-.09a1.4 1.4 0 0 1-.478-.252 1.1 1.1 0 0 1-.29-.375m-3.104-.033a1.3 1.3 0 0 1-.082-.466h.764a.6.6 0 0 0 .074.27.5.5 0 0 0 .454.246q.285 0 .422-.164.137-.165.137-.466v-2.745h.791v2.725q0 .66-.357 1.005-.355.345-.985.345a1.6 1.6 0 0 1-.568-.094 1.15 1.15 0 0 1-.407-.266 1.1 1.1 0 0 1-.243-.39m9.091-1.585v.522q0 .384-.117.641a.86.86 0 0 1-.322.387.9.9 0 0 1-.47.126.9.9 0 0 1-.47-.126.87.87 0 0 1-.32-.387 1.55 1.55 0 0 1-.117-.641v-.522q0-.386.117-.641a.87.87 0 0 1 .32-.387.87.87 0 0 1 .47-.129q.265 0 .47.129a.86.86 0 0 1 .322.387q.117.255.117.641m.803.519v-.513q0-.565-.205-.973a1.46 1.46 0 0 0-.59-.63q-.38-.22-.916-.22-.534 0-.92.22a1.44 1.44 0 0 0-.589.628q-.205.407-.205.975v.513q0 .562.205.973.205.407.589.626.386.217.92.217.536 0 .917-.217.384-.22.589-.626.204-.41.205-.973m1.29-.935v2.675h-.746v-3.999h.662l1.752 2.66h.032v-2.66h.75v4h-.656l-1.761-2.676z"/>
                                                    </svg>',
        'lock' => "<i class='bx bx-code-block'></i>",
        'tpl' => "<i class='bx bxl-html5'></i>",
        'css' => "<i class='bx bxs-file-css'></i>",
    ];
    $icon = $icons[$file['extension']] ?? "<i class='bx bx-folder'></i>";

    $html = '<tr data-path="' . htmlspecialchars($file['normal_path']) . '" data-name="' . htmlspecialchars($file['name']) . '">
        <td>
            <div id="content" class="d-flex align-items-center">
                <div>' . $icon . '</div>';

    if ($file['is_dir']) {
        $html .= '<div class="font-weight-bold"><a href="?action=file-management&path=' . htmlspecialchars($file['normal_path']) . '">
                    <i class="bi bi-folder"></i> &nbsp;' . htmlspecialchars($file['name']) . '
                </a></div>';
    } else {
        $html .= '<div class="font-weight-bold"><a href="' . htmlspecialchars($file['full_path']) . '">
                    <i class="bi bi-folder"></i> &nbsp;' . htmlspecialchars($file['name']) . '
                </a></div>';
    }

    $html .= '</div>
        </td>
        <td>'.htmlspecialchars($file['permissions']) .'</td>
        <td>' . htmlspecialchars($file['size']) . '</td>
        <td><i class="fa fa-ellipsis-h font-24"></i></td>
    </tr>';

    return $html;
}
